#!/usr/bin/env python3
"""
===============================================================================
   ⚓ BITCOIN ANCHOR TOOL (Public Edition)
   -----------------------------------------------------------------------
   A simple utility to timestamp files on the Bitcoin Blockchain.

   Author:  Anthro Entertainment LLC
   License: MIT License

   HOW IT WORKS:
   1. Hashes your file (SHA-256).
   2. Submits the hash to OpenTimestamps calendar servers.
   3. Receives a .ots proof file immediately.

   REQUIREMENTS:
   - pip install opentimestamps-client
===============================================================================
"""

import os
import sys
import json
import hashlib
import subprocess
import shutil
import glob
import readline
from datetime import datetime, timezone

# --- COLORS ---
C_RESET  = "\033[0m"
C_CYAN   = "\033[1;36m"
C_GREEN  = "\033[1;32m"
C_YELLOW = "\033[1;33m"
C_RED    = "\033[1;31m"
C_BOLD   = "\033[1m"
C_GREY   = "\033[90m"

# --- CONFIGURATION ---
HISTORY_FILE = "anchor_history.json"

# --- SETUP ---
def path_completer(text, state):
    return [x for x in glob.glob(text + '*')][state]

readline.set_completer_delims(' \t\n;')
readline.parse_and_bind("tab: complete")
readline.set_completer(path_completer)

OTS_EXEC = shutil.which("ots")
if not OTS_EXEC:
    # Fallback for local user installs
    user_bin = os.path.expanduser("~/.local/bin/ots")
    if os.path.exists(user_bin):
        OTS_EXEC = user_bin
    else:
        print(f"{C_RED}❌ Error: 'ots' command not found.{C_RESET}")
        print("   Please run: pip install opentimestamps-client")
        sys.exit(1)

def get_hash(filepath):
    print(f"   {C_CYAN}⚙️  Hashing {os.path.basename(filepath)}...{C_RESET}")
    sha256 = hashlib.sha256()
    file_size = os.path.getsize(filepath)
    processed = 0
    with open(filepath, "rb") as f:
        while chunk := f.read(16*1024*1024):
            sha256.update(chunk)
            processed += len(chunk)
            if file_size > 0:
                print(f"      Progress: {int((processed/file_size)*100)}%", end="\r")
    print("      Progress: 100%      ")
    return sha256.hexdigest()

def log_anchor(filename, file_hash, note):
    """Saves a local record of what you stamped."""
    data = {"records": []}
    if os.path.exists(HISTORY_FILE):
        try:
            with open(HISTORY_FILE, 'r') as f: data = json.load(f)
        except: pass

    record = {
        "filename": filename,
        "sha256": file_hash,
        "note": note,
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "proof_file": filename + ".ots"
    }

    data["records"].append(record)

    try:
        with open(HISTORY_FILE, 'w') as f:
            json.dump(data, f, indent=2)
    except: pass

def anchor_file(target_file):
    print("\n" + "="*60)
    print(f"{C_BOLD} ⚓ ANCHORING: {target_file}{C_RESET}")
    print("="*60)

    # 1. Check if proof already exists
    if os.path.exists(target_file + ".ots"):
        print(f"{C_YELLOW}⚠️  WARNING: A proof file already exists!{C_RESET}")
        print(f"   {target_file}.ots")
        confirm = input("   Overwrite? [y/N]: ").strip().lower()
        if confirm != 'y':
            print("   Cancelled.")
            return

    # 2. Hash it
    file_hash = get_hash(target_file)

    # 3. Optional Note
    note = input(f"   Enter a note for your local log (Optional) > ").strip()

    # 4. Stamp it
    print(f"   {C_CYAN}🚀 Submitting Hash to Bitcoin Calendar...{C_RESET}")
    try:
        # Calls 'ots stamp filename' which creates 'filename.ots' automatically
        subprocess.check_call([OTS_EXEC, "stamp", target_file])

        # 5. Log it
        log_anchor(os.path.basename(target_file), file_hash, note)

        print(f"\n{C_GREEN}   ✅ SUCCESS!{C_RESET}")
        print(f"      Proof Created: {C_BOLD}{target_file}.ots{C_RESET}")
        print(f"      Status: Pending (Wait ~10 mins for Bitcoin confirmation)")

    except subprocess.CalledProcessError as e:
        print(f"{C_RED}❌ Error during stamping: {e}{C_RESET}")

def main():
    print(f"\n{C_BOLD}🐾 FurryOS BITCOIN ANCHOR TOOL{C_RESET}")
    print("-" * 60)

    # List files
    # Exclude script itself, hidden files, and proof files from the main list
    files = sorted([f for f in os.listdir('.') if os.path.isfile(f) and not f.startswith('.')])

    has_files = False
    for f in files:
        if f.endswith('.ots') or f.endswith('.py') or f == HISTORY_FILE:
            continue

        has_files = True

        # Check for companion .ots file
        if os.path.exists(f + ".ots"):
            # USER REQUEST: Proof Found = YELLOW
            print(f" 📦 {f} {C_YELLOW}[Proof Found]{C_RESET}")
        else:
            # USER REQUEST: No Proof = GREY
            print(f" 📄 {f} {C_GREY}[Unstamped]{C_RESET}")

    if not has_files:
        print("   (No stampable files found in this directory)")

    print("-" * 60)

    try:
        target = input(f"\n{C_CYAN}   Enter filename to stamp > {C_RESET}").strip().strip("'")
        if target:
            if os.path.exists(target):
                anchor_file(target)
            else:
                print(f"{C_RED}   File not found.{C_RESET}")
    except: pass

if __name__ == "__main__":
    main()
